<?php
/**
 * Plugin Name: CtrlAltImran Professional Login
 * Description: Professional, clean custom WordPress login screen with logo, heading and description options. Affects only the login page.
 * Version: 1.0.1
 * Author: ctrlaltimran.com
 * Author URI: https://ctrlaltimran.com
 * License: GPL2+
 * Text Domain: ctrlaltimran-professional-login
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class CtrlAltImran_Professional_Login_Clean {

    private $option_name = 'ctrlaltimran_professional_login_clean_options';

    public function __construct() {
        // Admin settings (Tools tab)
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        // Login screen only
        add_action( 'login_enqueue_scripts', array( $this, 'custom_login_styles' ) );
        add_filter( 'login_headerurl', array( $this, 'custom_login_logo_url' ) );
        add_filter( 'login_headertext', array( $this, 'custom_login_logo_title' ) );
        add_filter( 'login_message', array( $this, 'custom_login_message' ) );
    }

    private function get_options() {
        $defaults = array(
            'logo_url'         => '',
            'heading_text'     => 'Welcome back',
            'description_text' => 'Sign in to access your dashboard',
            'background_color' => '#020617',
            'accent_color'     => '#6366f1',
            'card_background'  => '#020617',
        );

        $options = get_option( $this->option_name, array() );
        $options = wp_parse_args( $options, $defaults );

        // Simple color validation fallback
        if ( empty( $options['background_color'] ) || ! preg_match( '/^#([A-Fa-f0-9]{3}){1,2}$/', $options['background_color'] ) ) {
            $options['background_color'] = $defaults['background_color'];
        }
        if ( empty( $options['accent_color'] ) || ! preg_match( '/^#([A-Fa-f0-9]{3}){1,2}$/', $options['accent_color'] ) ) {
            $options['accent_color'] = $defaults['accent_color'];
        }
        if ( empty( $options['card_background'] ) || ! preg_match( '/^#([A-Fa-f0-9]{3}){1,2}$/', $options['card_background'] ) ) {
            $options['card_background'] = $defaults['card_background'];
        }

        return $options;
    }

    public function add_settings_page() {
        add_management_page(
            __( 'Professional Login', 'ctrlaltimran-professional-login' ),
            __( 'Professional Login', 'ctrlaltimran-professional-login' ),
            'manage_options',
            'ctrlaltimran-professional-login-clean',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting(
            'ctrlaltimran_professional_login_clean_group',
            $this->option_name,
            array( $this, 'sanitize_settings' )
        );

        add_settings_section(
            'ctrlaltimran_professional_login_clean_section_main',
            __( 'Login Page Branding', 'ctrlaltimran-professional-login' ),
            '__return_false',
            'ctrlaltimran_professional_login_clean'
        );

        add_settings_field(
            'logo_url',
            __( 'Logo URL', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_logo_url' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );

        add_settings_field(
            'heading_text',
            __( 'Heading', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_heading_text' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );

        add_settings_field(
            'description_text',
            __( 'Description', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_description_text' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );

        add_settings_field(
            'background_color',
            __( 'Background Color', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_background_color' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );

        add_settings_field(
            'accent_color',
            __( 'Accent Color', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_accent_color' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );

        add_settings_field(
            'card_background',
            __( 'Form Card Background', 'ctrlaltimran-professional-login' ),
            array( $this, 'field_card_background' ),
            'ctrlaltimran_professional_login_clean',
            'ctrlaltimran_professional_login_clean_section_main'
        );
    }

    public function sanitize_settings( $input ) {
        $output = array();

        $output['logo_url']         = isset( $input['logo_url'] ) ? esc_url_raw( $input['logo_url'] ) : '';
        $output['heading_text']     = isset( $input['heading_text'] ) ? sanitize_text_field( $input['heading_text'] ) : '';
        $output['description_text'] = isset( $input['description_text'] ) ? sanitize_textarea_field( $input['description_text'] ) : '';

        $bg   = isset( $input['background_color'] ) ? sanitize_hex_color( $input['background_color'] ) : '';
        $acc  = isset( $input['accent_color'] ) ? sanitize_hex_color( $input['accent_color'] ) : '';
        $card = isset( $input['card_background'] ) ? sanitize_hex_color( $input['card_background'] ) : '';

        $output['background_color'] = $bg ? $bg : '#020617';
        $output['accent_color']     = $acc ? $acc : '#6366f1';
        $output['card_background']  = $card ? $card : '#020617';

        return $output;
    }

    public function field_logo_url() {
        $options = $this->get_options();
        ?>
        <input type="url" name="<?php echo esc_attr( $this->option_name ); ?>[logo_url]" value="<?php echo esc_attr( $options['logo_url'] ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'Full URL of your logo image (PNG, SVG, or JPG).', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function field_heading_text() {
        $options = $this->get_options();
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[heading_text]" value="<?php echo esc_attr( $options['heading_text'] ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'Main heading shown above the login form.', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function field_description_text() {
        $options = $this->get_options();
        ?>
        <textarea name="<?php echo esc_attr( $this->option_name ); ?>[description_text]" rows="3" class="large-text"><?php echo esc_textarea( $options['description_text'] ); ?></textarea>
        <p class="description"><?php esc_html_e( 'Short description shown below the heading.', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function field_background_color() {
        $options = $this->get_options();
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[background_color]" value="<?php echo esc_attr( $options['background_color'] ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'Background color of the login screen (hex). Example: #020617', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function field_accent_color() {
        $options = $this->get_options();
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[accent_color]" value="<?php echo esc_attr( $options['accent_color'] ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'Primary accent color for buttons and links (hex). Example: #6366f1', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function field_card_background() {
        $options = $this->get_options();
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[card_background]" value="<?php echo esc_attr( $options['card_background'] ); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e( 'Background color of the login form card (hex).', 'ctrlaltimran-professional-login' ); ?></p>
        <?php
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Professional Login Settings', 'ctrlaltimran-professional-login' ); ?></h1>
            <p><?php esc_html_e( 'This plugin only affects the WordPress login screen. It does not change your admin area or front-end theme.', 'ctrlaltimran-professional-login' ); ?></p>
            <form method="post" action="options.php">
                <?php
                    settings_fields( 'ctrlaltimran_professional_login_clean_group' );
                    do_settings_sections( 'ctrlaltimran_professional_login_clean' );
                    submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function custom_login_styles() {
        $options = $this->get_options();

        $background_color = $options['background_color'];
        $accent_color     = $options['accent_color'];
        $card_background  = $options['card_background'];
        $logo_url         = $options['logo_url'];
        ?>
        <style type="text/css">
            body.login {
                background: radial-gradient(circle at top left, #1f2937 0, <?php echo esc_html( $background_color ); ?> 40%, #020617 100%);
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            }

            #login {
                width: 380px;
                padding-top: 40px;
            }

            .login h1 {
                margin-bottom: 20px;
            }

            .login h1 a,
#login h1 a,
body.login div#login h1 a,
body.login #login h1 a {
    background-image: url('<?php echo esc_url( $logo_url ); ?>') !important;
    background-size: contain !important;
    background-repeat: no-repeat !important;
    background-position: center !important;
    width: 260px !important;
    height: 120px !important;
    display: block !important;
    text-indent: -9999px !important;
    overflow: hidden !important;
    padding: 0 !important;
}


            .login form {
                background: <?php echo esc_html( $card_background ); ?>;
                border-radius: 18px;
                border: 1px solid rgba(148, 163, 184, 0.3);
                padding: 28px 28px 30px;
                box-shadow: 0 20px 50px rgba(15, 23, 42, 0.7);
            }

            .login label {
                color: black;
                font-weight: 500;
                font-size: 13px;
            }

            .login form .input,
            .login input[type="text"],
            .login input[type="password"],
            .login input[type="email"] {
                background: rgba(15, 23, 42, 0.96);
                border-radius: 999px;
                border: 1px solid rgba(148, 163, 184, 0.6);
                color: #e5e7eb;
                box-shadow: inset 0 1px 1px rgba(15, 23, 42, 0.9);
            }

            .login form .input:focus {
                border-color: <?php echo esc_html( $accent_color ); ?>;
                box-shadow:
                    0 0 0 1px <?php echo esc_html( $accent_color ); ?>,
                    0 0 0 4px rgba(99, 102, 241, 0.25);
            }

            .login .button-primary {
                width: 100%;
                border-radius: 999px;
                background: linear-gradient(135deg, <?php echo esc_html( $accent_color ); ?>, #22c55e);
                border: 0;
                text-shadow: 0 1px 1px rgba(15, 23, 42, 0.45);
                box-shadow: 0 12px 25px rgba(22, 163, 74, 0.45);
                padding: 0 18px;
                height: 40px;
                font-weight: 600;
                letter-spacing: 0.02em;
            }

            .login .button-primary:hover {
                filter: brightness(1.04);
                transform: translateY(-1px);
            }

            .login .button-primary:active {
                transform: translateY(0);
                box-shadow: 0 8px 18px rgba(22, 163, 74, 0.4);
            }

            .ctrlaltimran-login-heading {
                margin-bottom: 4px;
                font-size: 24px;
                font-weight: 700;
                letter-spacing: 0.02em;
                color: #f9fafb;
                text-align: left;
            }

            .ctrlaltimran-login-description {
                margin-bottom: 16px;
                font-size: 13px;
                color: #9ca3af;
                line-height: 1.5;
                text-align: left;
            }

            .login form .forgetmenot {
                display: flex;
                align-items: center;
            }

            .login form .forgetmenot input[type="checkbox"] + label {
                margin-left: 6px;
            }

            .login #nav,
            .login #backtoblog {
                text-align: center;
            }

            .login #nav a,
            .login #backtoblog a {
                color: #9ca3af;
                font-size: 12px;
            }

            .login #nav a:hover,
            .login #backtoblog a:hover {
                color: <?php echo esc_html( $accent_color ); ?>;
            }

            .login #login_error,
            .login .message {
                border-radius: 14px;
                border: 1px solid rgba(248, 113, 113, 0.4);
                background: rgba(30, 64, 175, 0.18);
                color: #fee2e2;
                box-shadow: 0 12px 30px rgba(15, 23, 42, 0.7);
            }

            .login a {
                color: <?php echo esc_html( $accent_color ); ?>;
            }

            .login a:hover {
                color: #22c55e;
            }
        </style>
        <?php
    }

    public function custom_login_logo_url() {
        // Keep user on site home, no WordPress.org
        return home_url();
    }

    public function custom_login_logo_title( $title ) {
        $options = $this->get_options();
        if ( ! empty( $options['heading_text'] ) ) {
            return $options['heading_text'];
        }
        return get_bloginfo( 'name' );
    }

    public function custom_login_message( $message ) {
        $options = $this->get_options();

        $heading     = ! empty( $options['heading_text'] ) ? $options['heading_text'] : '';
        $description = ! empty( $options['description_text'] ) ? $options['description_text'] : '';

        $output  = '<div class="ctrlaltimran-login-heading">' . esc_html( $heading ) . '</div>';
        if ( ! empty( $description ) ) {
            $output .= '<div class="ctrlaltimran-login-description">' . esc_html( $description ) . '</div>';
        }

        // Prepend our message, keep any core messages (errors etc)
        return $output . $message;
    }
}

new CtrlAltImran_Professional_Login_Clean();
