<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class CAAI_Meta_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
    }

    public function add_settings_page() {
        add_options_page(
            'CtrlAlt AI Meta',
            'CtrlAlt AI Meta',
            'manage_options',
            'caai-meta',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting(
            'caai_meta_group',
            'caai_meta_api_key',
            array(
                'type'              => 'string',
                'sanitize_callback' => array( $this, 'sanitize_api_key' ),
                'default'           => '',
            )
        );

        register_setting(
            'caai_meta_group',
            'caai_meta_post_types',
            array(
                'type'              => 'array',
                'sanitize_callback' => array( $this, 'sanitize_post_types' ),
                'default'           => array( 'page' ),
            )
        );
    }

    public function sanitize_api_key( $value ) {
        $value = trim( $value );
        return $value;
    }

    public function sanitize_post_types( $value ) {
        if ( ! is_array( $value ) ) {
            return array( 'page' );
        }
        $clean = array();
        foreach ( $value as $pt ) {
            $pt = sanitize_key( $pt );
            if ( in_array( $pt, array( 'post', 'page' ), true ) ) {
                $clean[] = $pt;
            }
        }
        if ( empty( $clean ) ) {
            $clean = array( 'page' );
        }
        return $clean;
    }

    public function enqueue_assets( $hook ) {
        if ( $hook !== 'settings_page_caai-meta' ) {
            return;
        }

        wp_enqueue_script(
            'caai-meta-admin',
            CAAI_META_URL . 'assets/admin.js',
            array( 'jquery' ),
            CAAI_META_VERSION,
            true
        );

        wp_localize_script(
            'caai-meta-admin',
            'CAAI_Meta',
            array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'caai_meta_nonce' ),
            )
        );

        wp_enqueue_style(
            'caai-meta-admin-css',
            CAAI_META_URL . 'assets/admin.css',
            array(),
            CAAI_META_VERSION
        );
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $api_key    = get_option( 'caai_meta_api_key', '' );
        $post_types = get_option( 'caai_meta_post_types', array( 'page' ) );
        if ( ! is_array( $post_types ) ) {
            $post_types = array( 'page' );
        }
        ?>
        <div class="wrap">
            <h1>CtrlAlt AI Meta Descriptions</h1>
            <p>
                This tool uses OpenAI to generate clean, SEO-friendly meta descriptions for your pages and posts.
                Descriptions are saved into the Yoast SEO meta description field.
            </p>
            <p><strong>Tip:</strong> Always take a database backup before running bulk actions on a live site.</p>

            <form method="post" action="options.php" class="caai-meta-settings-form">
                <?php
                settings_fields( 'caai_meta_group' );
                ?>
                <h2 class="title">Settings</h2>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="caai_meta_api_key">OpenAI API key</label></th>
                        <td>
                            <input type="password" name="caai_meta_api_key" id="caai_meta_api_key" class="regular-text" value="<?php echo esc_attr( $api_key ); ?>" autocomplete="off" />
                            <p class="description">Your secret OpenAI API key. Stored in the database as plain text, so keep it safe.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Post types to include</th>
                        <td>
                            <label>
                                <input type="checkbox" name="caai_meta_post_types[]" value="page" <?php checked( in_array( 'page', $post_types, true ) ); ?> />
                                Pages
                            </label>
                            <br />
                            <label>
                                <input type="checkbox" name="caai_meta_post_types[]" value="post" <?php checked( in_array( 'post', $post_types, true ) ); ?> />
                                Posts
                            </label>
                            <p class="description">Choose which content types will be processed in the bulk generator.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( 'Save Settings' ); ?>
            </form>

            <hr />

            <h2>Bulk generate meta descriptions</h2>
            <p>
                This will scan the selected post types, send each one to OpenAI, and write back a fresh meta description.
                Existing Yoast descriptions will be overwritten.
            </p>
            <p><strong>Make sure your API key is valid and you understand token usage and costs on your OpenAI account.</strong></p>

            <button id="caai-meta-start" class="button button-primary button-hero">Run bulk generation</button>

            <div id="caai-meta-progress-wrapper" style="display:none;margin-top:20px;">
                <div id="caai-meta-progress-text">Preparing…</div>
                <div class="caai-meta-progress-bar-outer">
                    <div id="caai-meta-progress-bar-inner"></div>
                </div>
                <ul id="caai-meta-log"></ul>
            </div>
        </div>
        <?php
    }
}
