<?php
/**
 * Plugin Name: CtrlAlt AI Meta Descriptions
 * Description: Generate clean SEO meta descriptions for pages and posts using OpenAI. Includes bulk generate with progress log. Writes to Yoast SEO meta description field.
 * Version: 1.1.0
 * Author: CtrlAltImran
 * License: GPLv2 or later
 * Requires at least: 5.5
 * Requires PHP: 7.2
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'CAAI_META_VERSION', '1.1.0' );
define( 'CAAI_META_PATH', plugin_dir_path( __FILE__ ) );
define( 'CAAI_META_URL', plugin_dir_url( __FILE__ ) );

require_once CAAI_META_PATH . 'includes/class-caai-meta-admin.php';

class CAAI_Meta_Core {

    public function __construct() {
        new CAAI_Meta_Admin();

        add_action( 'wp_ajax_caai_meta_get_targets', array( $this, 'ajax_get_targets' ) );
        add_action( 'wp_ajax_caai_meta_generate_for_post', array( $this, 'ajax_generate_for_post' ) );
    }

    /**
     * AJAX: Get list of target post IDs based on settings.
     */
    public function ajax_get_targets() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions.' ) );
        }

        check_ajax_referer( 'caai_meta_nonce', 'nonce' );

        $post_types = get_option( 'caai_meta_post_types', array( 'page' ) );

        if ( ! is_array( $post_types ) || empty( $post_types ) ) {
            $post_types = array( 'page' );
        }

        $post_types = array_values( array_unique( array_map( 'sanitize_text_field', $post_types ) ) );

        $args = array(
            'post_type'      => $post_types,
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => -1,
        );

        $q   = new WP_Query( $args );
        $ids = $q->posts;

        wp_send_json_success( array(
            'ids'   => $ids,
            'count' => count( $ids ),
        ) );
    }

    /**
     * Call OpenAI with simple retry for 429 rate limit.
     */
    protected function caai_call_openai( $body, $api_key ) {
        $max_attempts = 2;
        $delay        = 5; // seconds
        $attempt      = 0;
        $last_response = null;

        while ( $attempt < $max_attempts ) {
            $attempt++;

            $response = wp_remote_post(
                'https://api.openai.com/v1/chat/completions',
                array(
                    'headers' => array(
                        'Content-Type'  => 'application/json',
                        'Authorization' => 'Bearer ' . $api_key,
                    ),
                    'body'    => wp_json_encode( $body ),
                    'timeout' => 45,
                )
            );

            if ( is_wp_error( $response ) ) {
                return $response;
            }

            $code = wp_remote_retrieve_response_code( $response );

            if ( 429 === intval( $code ) && $attempt < $max_attempts ) {
                // Rate limited, wait then retry.
                sleep( $delay );
                $last_response = $response;
                continue;
            }

            return $response;
        }

        return $last_response;
    }

    /**
     * AJAX: Generate meta description for a single post and save to Yoast field.
     */
    public function ajax_generate_for_post() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions.' ) );
        }

        check_ajax_referer( 'caai_meta_nonce', 'nonce' );

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        if ( ! $post_id || get_post_status( $post_id ) !== 'publish' ) {
            wp_send_json_error( array( 'message' => 'Invalid post ID.' ) );
        }

        $api_key = trim( get_option( 'caai_meta_api_key', '' ) );
        if ( empty( $api_key ) ) {
            wp_send_json_error( array( 'message' => 'OpenAI API key is not set. Please add it in Settings → CtrlAlt AI Meta.' ) );
        }

        $post = get_post( $post_id );
        if ( ! $post ) {
            wp_send_json_error( array( 'message' => 'Post not found.' ) );
        }

        $title   = get_the_title( $post );
        $content = apply_filters( 'the_content', $post->post_content );
        $content = wp_strip_all_tags( $content );
        $content = preg_replace( '/\s+/', ' ', $content );
        if ( function_exists( 'mb_substr' ) ) {
            $content = mb_substr( $content, 0, 2000 );
        } else {
            $content = substr( $content, 0, 2000 );
        }

        $site_name = get_bloginfo( 'name' );
        $prompt  = "You are an SEO copywriter.\n";
        $prompt .= "Write a natural, compelling meta description for this page. It must be:\n";
        $prompt .= "- Maximum 155 characters.\n";
        $prompt .= "- Focused on benefits for the visitor.\n";
        $prompt .= "- In neutral, professional tone.\n";
        $prompt .= "- Do NOT use quotation marks.\n\n";
        $prompt .= "Site name: " . $site_name . "\n";
        $prompt .= "Page title: " . $title . "\n\n";
        $prompt .= "Page content:\n" . $content . "\n";

        $body = array(
            'model'    => 'gpt-4o-mini',
            'messages' => array(
                array(
                    'role'    => 'system',
                    'content' => 'You write concise, high-quality SEO meta descriptions.',
                ),
                array(
                    'role'    => 'user',
                    'content' => $prompt,
                ),
            ),
            'temperature' => 0.7,
            'max_tokens'  => 120,
        );

        $response = $this->caai_call_openai( $body, $api_key );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => 'API error: ' . $response->get_error_message() ) );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $raw  = wp_remote_retrieve_body( $response );
        if ( 200 !== intval( $code ) || empty( $raw ) ) {
            wp_send_json_error( array( 'message' => 'API HTTP error. Code: ' . $code ) );
        }

        $data = json_decode( $raw, true );
        if ( ! isset( $data['choices'][0]['message']['content'] ) ) {
            wp_send_json_error( array( 'message' => 'Unexpected API response.' ) );
        }

        $desc = trim( $data['choices'][0]['message']['content'] );
        // Remove surrounding quotes if any.
        $desc = preg_replace( '/^"+|"+$/', '', $desc );
        $desc = preg_replace( "/^'+|'+$/", '', $desc );

        if ( function_exists( 'mb_substr' ) ) {
            $desc = mb_substr( $desc, 0, 160 );
        } else {
            $desc = substr( $desc, 0, 160 );
        }

        update_post_meta( $post_id, '_yoast_wpseo_metadesc', wp_kses_post( $desc ) );

        wp_send_json_success( array(
            'post_id'     => $post_id,
            'title'       => $title,
            'description' => $desc,
        ) );
    }
}

new CAAI_Meta_Core();
